import hashlib
import json
import requests
from datetime import datetime
from typing import List, Dict, Any

class EmptyNumberBatchDetector:
    def __init__(self, hashcode: str, private_key: str):
        self.api_url = "http://ip:port/HrmApi/emptyStatusQuery_batch"
        self.hashcode = hashcode
        self.private_key = private_key
    
    def _generate_sign(self, date: str) -> str:
        sign_str = f"{self.hashcode}{self.private_key}{date}"
        return hashlib.md5(sign_str.encode('utf-8')).hexdigest()
    
    def _send_request(self, mobiles: List[str]) -> Dict[str, Any]:
        date = datetime.now().strftime('%Y%m%d')
        sign = self._generate_sign(date)
        
        payload = {
            "mobiles": mobiles,
            "hashcode": self.hashcode,
            "sign": sign
        }
        
        headers = {'Content-Type': 'application/json'}
        response = requests.post(self.api_url, data=json.dumps(payload), headers=headers)
        return response.json()
    
    def detect(self, phone_numbers: List[str]) -> List[Dict[str, Any]]:
        results = []
        chunk_size = 1000
        
        for i in range(0, len(phone_numbers), chunk_size):
            chunk = phone_numbers[i:i + chunk_size]
            response = self._send_request(chunk)
            
            if response.get('errorRes', {}).get('err_code') == 200:
                results.extend(response.get('mobileresult', []))
            else:
                raise Exception(f"API Error: {response.get('errorRes', {}).get('err_content', 'Unknown error')}")
        
        return results

# 使用示例
if __name__ == "__main__":
    detector = EmptyNumberBatchDetector("your_hashcode", "your_private_key")
    phone_numbers = ["13111111111", "13222222222"]  # 手机号列表
    
    try:
        results = detector.detect(phone_numbers)
        print(results)
    except Exception as e:
        print(f"Error: {str(e)}")